<?php
/**
 * Admin Settings Page
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Tool_Admin {
    private $auth;
    
    public function __construct() {
        $this->auth = new SEO_Tool_Auth();
    }
    
    public function init() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_init', [$this, 'handle_disconnect']);
        add_action('admin_init', [$this, 'handle_save_key']);
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'SEO Tool',
            'SEO Tool',
            'manage_options',
            'seo-tool',
            [$this, 'render_settings_page'],
            'dashicons-admin-links',
            30
        );
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts($hook) {
        if ($hook !== 'toplevel_page_seo-tool') {
            return;
        }
        
        wp_enqueue_style(
            'seo-tool-admin',
            SEO_TOOL_PLUGIN_URL . 'admin/assets/css/admin.css',
            [],
            SEO_TOOL_VERSION
        );
        
        wp_enqueue_script(
            'seo-tool-admin',
            SEO_TOOL_PLUGIN_URL . 'admin/assets/js/admin.js',
            ['jquery'],
            SEO_TOOL_VERSION,
            true
        );
    }
    
    /**
     * Handle disconnect
     */
    public function handle_disconnect() {
        if (!isset($_POST['seo_tool_disconnect']) || !check_admin_referer('seo_tool_disconnect')) {
            return;
        }
        
        $this->auth->disconnect();
        wp_redirect(admin_url('admin.php?page=seo-tool&disconnected=1'));
        exit;
    }
    
    /**
     * Handle integration key save
     */
    public function handle_save_key() {
        if (!isset($_POST['seo_tool_save_key']) || !check_admin_referer('seo_tool_save_key')) {
            return;
        }
        
        $key = isset($_POST['integration_key']) ? sanitize_text_field($_POST['integration_key']) : '';
        $post_mode = isset($_POST['post_mode']) ? sanitize_text_field($_POST['post_mode']) : 'publish';
        
        if (empty($key)) {
            wp_redirect(admin_url('admin.php?page=seo-tool&error=empty_key'));
            exit;
        }
        
        $result = $this->auth->validate_integration_key($key);
        
        if ($result['success']) {
            update_option('seo_tool_integration_key', $key);
            update_option('seo_tool_post_mode', $post_mode);
            update_option('seo_tool_connected', true);
            update_option('seo_tool_connected_at', current_time('mysql'));
            update_option('seo_tool_user_id', $result['user_id']);
            wp_redirect(admin_url('admin.php?page=seo-tool&connected=1'));
            exit;
        } else {
            wp_redirect(admin_url('admin.php?page=seo-tool&error=' . urlencode($result['message'])));
            exit;
        }
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        $is_connected = $this->auth->is_connected();
        $connected_at = get_option('seo_tool_connected_at');
        $user_id = get_option('seo_tool_user_id');
        $last_publish = get_option('seo_tool_last_publish');
        $total_published = get_option('seo_tool_total_published', 0);
        $integration_key = get_option('seo_tool_integration_key', '');
        $post_mode = get_option('seo_tool_post_mode', 'publish');
        $error_message = isset($_GET['error']) ? urldecode($_GET['error']) : '';
        
        ?>
        <div class="wrap seo-tool-settings">
            <h1>SEO Tool Integration</h1>
            
            <?php if (isset($_GET['connected'])): ?>
                <div class="notice notice-success is-dismissible">
                    <p>Successfully connected to SEO Tool!</p>
                </div>
            <?php endif; ?>
            
            <?php if (isset($_GET['disconnected'])): ?>
                <div class="notice notice-info is-dismissible">
                    <p>Disconnected from SEO Tool.</p>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo esc_html($error_message); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (!$is_connected): ?>
                <div class="card">
                    <h2>Connect to SEO Tool</h2>
                    
                    <!-- Integration Key Method -->
                    <div style="margin-bottom: 30px;">
                        <h3>Method 1: Integration Key (Recommended)</h3>
                        <p>Generate an integration key in your SEO Tool dashboard and paste it here.</p>
                        
                        <form method="post" action="">
                            <?php wp_nonce_field('seo_tool_save_key'); ?>
                            <table class="form-table">
                                <tr>
                                    <th>
                                        <label for="integration_key">Integration Key</label>
                                    </th>
                                    <td>
                                        <input 
                                            type="text" 
                                            id="integration_key" 
                                            name="integration_key" 
                                            value="<?php echo esc_attr($integration_key); ?>"
                                            class="regular-text"
                                            placeholder="wp_key_xxxx-xxxx-xxxx"
                                        />
                                        <p class="description">
                                            Generate this key in <strong>SEO Tool</strong> when creating your WordPress integration.
                                        </p>
                                    </td>
                                </tr>
                                <tr>
                                    <th>
                                        <label for="post_mode">Post Mode</label>
                                    </th>
                                    <td>
                                        <select id="post_mode" name="post_mode" class="regular-text">
                                            <option value="publish" <?php selected($post_mode, 'publish'); ?>>Publish Directly</option>
                                            <option value="draft" <?php selected($post_mode, 'draft'); ?>>Save as Draft</option>
                                        </select>
                                        <p class="description">
                                            Choose whether incoming posts are published immediately or saved as drafts.
                                        </p>
                                    </td>
                                </tr>
                            </table>
                            <p class="submit">
                                <input type="submit" name="seo_tool_save_key" class="button button-primary" value="Save" />
                            </p>
                        </form>
                    </div>
                    
                    <!-- OAuth Method (Alternative) -->
                    <div>
                        <h3>Method 2: OAuth Connection</h3>
                        <p>Connect via OAuth redirect flow (requires redirects).</p>
                        <a href="<?php echo esc_url($this->auth->get_authorization_url()); ?>" 
                           class="button button-secondary">
                            Connect via OAuth
                        </a>
                    </div>
                </div>
            <?php else: ?>
                <div class="card">
                    <h2>✅ Connected</h2>
                    <p>Your WordPress site is connected to your SEO Tool account.</p>
                    
                    <table class="form-table">
                        <tr>
                            <th>Status</th>
                            <td>
                                <span style="color: green;">●</span> Connected
                                <br>
                                <small>Connected at: <?php echo esc_html($connected_at); ?></small>
                            </td>
                        </tr>
                        <tr>
                            <th>Account</th>
                            <td>User ID: <?php echo esc_html($user_id); ?></td>
                        </tr>
                        <tr>
                            <th>Statistics</th>
                            <td>
                                Total published: <strong><?php echo esc_html($total_published); ?></strong>
                                <?php if ($last_publish): ?>
                                    <br>
                                    Last publish: <?php echo esc_html($last_publish); ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                    
                    <form method="post" action="">
                        <?php wp_nonce_field('seo_tool_disconnect'); ?>
                        <input type="hidden" name="seo_tool_disconnect" value="1">
                        <button type="submit" class="button button-secondary" onclick="return confirm('Are you sure you want to disconnect?');">
                            Disconnect
                        </button>
                    </form>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}

