<?php
/**
 * Custom REST API Endpoints
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Tool_API {
    private $auth;
    
    public function __construct() {
        $this->auth = new SEO_Tool_Auth();
    }
    
    public function init() {
        add_action('rest_api_init', [$this, 'register_routes']);
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        register_rest_route('seo-tool/v1', '/publish', [
            'methods' => 'POST',
            'callback' => [$this, 'publish_post'],
            'permission_callback' => [$this, 'verify_token'],
        ]);
        
        register_rest_route('seo-tool/v1', '/status', [
            'methods' => 'GET',
            'callback' => [$this, 'get_status'],
            'permission_callback' => [$this, 'verify_token'],
        ]);
        
        register_rest_route('seo-tool/v1', '/test', [
            'methods' => 'GET',
            'callback' => [$this, 'test_connection'],
            'permission_callback' => [$this, 'verify_token'],
        ]);
    }
    
    /**
     * Verify access token
     */
    public function verify_token($request) {
        $auth_header = $request->get_header('Authorization');
        
        if (!$auth_header) {
            return new WP_Error('missing_token', 'Authorization header required', ['status' => 401]);
        }
        
        // Extract Bearer token
        if (preg_match('/Bearer\s+(.*)$/i', $auth_header, $matches)) {
            $token = $matches[1];
        } else {
            return new WP_Error('invalid_token', 'Invalid token format', ['status' => 401]);
        }
        
        // Verify token
        if (!$this->auth->verify_token($token)) {
            return new WP_Error('invalid_token', 'Invalid or expired access token', ['status' => 401]);
        }
        
        return true;
    }
    
    /**
     * Publish post endpoint
     */
    public function publish_post($request) {
        $publisher = new SEO_Tool_Publisher();
        return $publisher->publish($request->get_json_params());
    }
    
    /**
     * Get status endpoint
     */
    public function get_status($request) {
        return [
            'plugin_version' => SEO_TOOL_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'api_key_active' => $this->auth->is_connected(),
            'connection_status' => $this->auth->is_connected() ? 'connected' : 'disconnected',
            'last_publish' => get_option('seo_tool_last_publish'),
            'total_published' => (int) get_option('seo_tool_total_published', 0),
        ];
    }
    
    /**
     * Test connection endpoint
     */
    public function test_connection($request) {
        return [
            'success' => true,
            'message' => 'Connection successful',
            'timestamp' => current_time('c'),
        ];
    }
}

