<?php
/**
 * OAuth Authentication Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Tool_Auth {
    private $api_base_url;
    
    public function __construct() {
        $this->api_base_url = get_option('seo_tool_api_base_url', 'https://seotoolstaging.site/api/v1');
    }
    
    public function init() {
        // Handle OAuth callback
        add_action('admin_init', [$this, 'handle_callback']);
    }
    
    /**
     * Get authorization URL
     */
    public function get_authorization_url() {
        $state = wp_generate_password(32, false);
        set_transient('seo_tool_oauth_state', $state, 600); // 10 min
        
        $params = [
            'response_type' => 'code',
            'client_id' => 'wordpress_plugin',
            'redirect_uri' => $this->get_redirect_uri(),
            'state' => $state,
            'scope' => 'publish read'
        ];
        
        return $this->api_base_url . '/auth/wordpress/authorize?' . http_build_query($params);
    }
    
    /**
     * Handle OAuth callback
     */
    public function handle_callback() {
        if (!isset($_GET['page']) || $_GET['page'] !== 'seo-tool') {
            return;
        }
        
        if (!isset($_GET['action']) || $_GET['action'] !== 'callback') {
            return;
        }
        
        if (!isset($_GET['code']) || !isset($_GET['state'])) {
            wp_die('Missing authorization code or state parameter.');
        }
        
        $code = sanitize_text_field($_GET['code']);
        $state = sanitize_text_field($_GET['state']);
        
        // Verify state
        $saved_state = get_transient('seo_tool_oauth_state');
        if ($state !== $saved_state) {
            wp_die('Invalid state parameter. Please try again.');
        }
        
        delete_transient('seo_tool_oauth_state');
        
        // Exchange code for token
        $response = wp_remote_post($this->api_base_url . '/auth/wordpress/token', [
            'body' => json_encode([
                'code' => $code,
                'redirect_uri' => $this->get_redirect_uri(),
                'grant_type' => 'authorization_code'
            ]),
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'timeout' => 30
        ]);
        
        if (is_wp_error($response)) {
            wp_die('Failed to connect: ' . $response->get_error_message());
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['access_token'])) {
            // Store tokens securely
            update_option('seo_tool_access_token', $body['access_token']);
            update_option('seo_tool_refresh_token', $body['refresh_token']);
            update_option('seo_tool_token_expires', time() + $body['expires_in']);
            update_option('seo_tool_user_id', $body['user_id']);
            update_option('seo_tool_connected', true);
            update_option('seo_tool_connected_at', current_time('mysql'));
            
            // Redirect to settings page with success message
            wp_redirect(admin_url('admin.php?page=seo-tool&connected=1'));
            exit;
        } else {
            wp_die('Failed to obtain access token. Please try again.');
        }
    }
    
    /**
     * Get access token (refresh if needed)
     */
    public function get_access_token() {
        $token = get_option('seo_tool_access_token');
        $expires = get_option('seo_tool_token_expires', 0);
        
        // Refresh if expired or expiring soon (within 5 minutes)
        if (time() >= ($expires - 300)) {
            return $this->refresh_token();
        }
        
        return $token;
    }
    
    /**
     * Refresh access token
     */
    private function refresh_token() {
        $refresh_token = get_option('seo_tool_refresh_token');
        
        if (!$refresh_token) {
            return false;
        }
        
        $response = wp_remote_post($this->api_base_url . '/auth/wordpress/refresh', [
            'body' => json_encode([
                'refresh_token' => $refresh_token,
                'grant_type' => 'refresh_token'
            ]),
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'timeout' => 30
        ]);
        
        if (is_wp_error($response)) {
            $this->disconnect();
            return false;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['access_token'])) {
            update_option('seo_tool_access_token', $body['access_token']);
            update_option('seo_tool_token_expires', time() + $body['expires_in']);
            return $body['access_token'];
        }
        
        // Refresh failed, disconnect
        $this->disconnect();
        return false;
    }
    
    /**
     * Disconnect from SEO Tool
     */
    public function disconnect() {
        delete_option('seo_tool_access_token');
        delete_option('seo_tool_refresh_token');
        delete_option('seo_tool_token_expires');
        delete_option('seo_tool_user_id');
        delete_option('seo_tool_connected');
        delete_option('seo_tool_connected_at');
        delete_option('seo_tool_webhook_url');
    }
    
    /**
     * Check if connected
     */
    public function is_connected() {
        $connected = get_option('seo_tool_connected', false);
        $token = $this->get_access_token();
        return $connected && !empty($token);
    }
    
    /**
     * Get redirect URI
     */
    private function get_redirect_uri() {
        return admin_url('admin.php?page=seo-tool&action=callback');
    }
    
    /**
     * Verify access token for API requests
     */
    public function verify_token($token) {
        $stored_token = $this->get_access_token();
        return $token === $stored_token;
    }
    
    /**
     * Validate integration key and get access token
     */
    public function validate_integration_key($integration_key) {
        $url = $this->api_base_url . '/auth/wordpress/validate-key';
        
        // Log the request for debugging
        error_log('[SEO Tool] Validating key: ' . substr($integration_key, 0, 20) . '...');
        error_log('[SEO Tool] API URL: ' . $url);
        error_log('[SEO Tool] WordPress Site URL: ' . home_url());
        
        $response = wp_remote_post($url, [
            'body' => json_encode([
                'integrationKey' => $integration_key,
                'wordpressSiteUrl' => home_url(),
            ]),
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'timeout' => 30,
            'sslverify' => true // Enable SSL verification for production
        ]);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('[SEO Tool] WP Error: ' . $error_message);
            return [
                'success' => false,
                'message' => 'Connection failed: ' . $error_message . '. Check if backend is running at ' . $this->api_base_url
            ];
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        error_log('[SEO Tool] Response Code: ' . $response_code);
        error_log('[SEO Tool] Response Body: ' . substr($response_body, 0, 500));
        
        if ($response_code !== 200) {
            return [
                'success' => false,
                'message' => 'Backend returned error ' . $response_code . '. Check backend logs.'
            ];
        }
        
        $body = json_decode($response_body, true);
        
        if (!$body) {
            error_log('[SEO Tool] Failed to parse JSON response');
            return [
                'success' => false,
                'message' => 'Invalid response from backend. Response: ' . substr($response_body, 0, 200)
            ];
        }
        
        if (isset($body['data']['access_token'])) {
            // Store tokens
            update_option('seo_tool_access_token', $body['data']['access_token']);
            update_option('seo_tool_refresh_token', $body['data']['refresh_token']);
            update_option('seo_tool_token_expires', time() + $body['data']['expires_in']);
            update_option('seo_tool_user_id', $body['data']['user_id']);
            
            error_log('[SEO Tool] ✅ Key validated successfully');
            
            return [
                'success' => true,
                'user_id' => $body['data']['user_id']
            ];
        }
        
        // Check for error message in response
        $error_msg = $body['message'] ?? 'Failed to validate integration key';
        if (isset($body['error'])) {
            $error_msg .= ': ' . (is_array($body['error']) ? $body['error']['message'] : $body['error']);
        }
        
        error_log('[SEO Tool] Validation failed: ' . $error_msg);
        
        return [
            'success' => false,
            'message' => $error_msg
        ];
    }
}

