<?php
/**
 * Publishing Logic
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Tool_Publisher {
    private $webhook;
    
    public function __construct() {
        $this->webhook = new SEO_Tool_Webhook();
    }
    
    /**
     * Publish post
     */
    public function publish($data) {
        try {
            // Validate required fields
            if (empty($data['title']) || empty($data['content'])) {
                return new WP_Error('missing_fields', 'Title and content are required', ['status' => 400]);
            }
            
            // Check for existing post with same seo_tool_blog_id
            if (!empty($data['meta_fields']['seo_tool_blog_id'])) {
                $existing_posts = get_posts([
                    'meta_key' => 'seo_tool_blog_id',
                    'meta_value' => $data['meta_fields']['seo_tool_blog_id'],
                    'post_type' => 'post',
                    'posts_per_page' => 1,
                    'post_status' => 'any',
                ]);
                
                if (!empty($existing_posts)) {
                    $existing_post = $existing_posts[0];
                    return [
                        'success' => true,
                        'post_id' => $existing_post->ID,
                        'post_url' => get_permalink($existing_post->ID),
                        'edit_url' => admin_url("post.php?post={$existing_post->ID}&action=edit"),
                        'status' => $existing_post->post_status,
                        'message' => 'Post already exists - duplicate prevented',
                        'duplicate' => true,
                    ];
                }
            }
            
            // Extract JSON-LD script tags from content before sanitization
            $json_ld_scripts = $this->extract_json_ld_scripts($data['content']);
            
            // Remove JSON-LD scripts from content before sanitization
            $content_without_json_ld = $this->remove_json_ld_scripts($data['content']);
            
            // Prepare post data
            $post_data = [
                'post_title' => sanitize_text_field($data['title']),
                'post_content' => wp_kses_post($content_without_json_ld),
                'post_excerpt' => isset($data['excerpt']) ? sanitize_textarea_field($data['excerpt']) : '',
                'post_name' => isset($data['slug']) ? sanitize_title($data['slug']) : '',
                'post_status' => isset($data['status']) ? sanitize_text_field($data['status']) : 'publish',
                'post_type' => 'post',
            ];
            
            // Handle publish date
            if (isset($data['publish_date'])) {
                $post_data['post_date'] = date('Y-m-d H:i:s', strtotime($data['publish_date']));
                $post_data['post_date_gmt'] = get_gmt_from_date($post_data['post_date']);
            }
            
            // Create post
            $post_id = wp_insert_post($post_data, true);
            
            if (is_wp_error($post_id)) {
                $this->send_webhook('post.failed', [
                    'error' => $post_id->get_error_message(),
                    'data' => $data,
                ]);
                return $post_id;
            }
            
            // Store JSON-LD scripts in post meta for later injection into head
            if (!empty($json_ld_scripts)) {
                update_post_meta($post_id, '_seo_tool_json_ld', $json_ld_scripts);
            }
            
            // Handle featured image
            if (!empty($data['featured_media_url'])) {
                $this->set_featured_image($post_id, $data['featured_media_url']);
            }
            
            // Handle categories
            if (!empty($data['categories'])) {
                wp_set_post_categories($post_id, $this->get_or_create_categories($data['categories']));
            }
            
            // Handle tags
            if (!empty($data['tags'])) {
                wp_set_post_tags($post_id, $data['tags']);
            }
            
            // Store SEO Tool metadata
            if (!empty($data['meta_fields'])) {
                foreach ($data['meta_fields'] as $key => $value) {
                    update_post_meta($post_id, sanitize_key($key), sanitize_meta($key, $value, 'post'));
                }
            }
            
            // Update statistics
            update_option('seo_tool_last_publish', current_time('mysql'));
            $total = (int) get_option('seo_tool_total_published', 0);
            update_option('seo_tool_total_published', $total + 1);
            
            // Send success webhook
            $this->send_webhook('post.published', [
                'post_id' => $post_id,
                'post_url' => get_permalink($post_id),
                'edit_url' => admin_url("post.php?post={$post_id}&action=edit"),
                'status' => get_post_status($post_id),
                'seo_tool_blog_id' => $data['meta_fields']['seo_tool_blog_id'] ?? null,
            ]);
            
            return [
                'success' => true,
                'post_id' => $post_id,
                'post_url' => get_permalink($post_id),
                'edit_url' => admin_url("post.php?post={$post_id}&action=edit"),
                'status' => get_post_status($post_id),
            ];
            
        } catch (Exception $e) {
            $this->send_webhook('post.failed', [
                'error' => $e->getMessage(),
                'data' => $data,
            ]);
            
            return new WP_Error('publish_failed', $e->getMessage(), ['status' => 500]);
        }
    }
    
    /**
     * Set featured image
     */
    private function set_featured_image($post_id, $image_url) {
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        
        // Download image
        $tmp = download_url($image_url);
        
        if (is_wp_error($tmp)) {
            return;
        }
        
        // Prepare file array
        $file_array = [
            'name' => basename($image_url),
            'tmp_name' => $tmp,
        ];
        
        // Upload file
        $attachment_id = media_handle_sideload($file_array, $post_id);
        
        if (is_wp_error($attachment_id)) {
            @unlink($tmp);
            return;
        }
        
        // Set as featured image
        set_post_thumbnail($post_id, $attachment_id);
    }
    
    /**
     * Get or create categories
     */
    private function get_or_create_categories($category_names) {
        $category_ids = [];
        
        foreach ($category_names as $name) {
            $term = get_term_by('name', $name, 'category');
            
            if ($term) {
                $category_ids[] = $term->term_id;
            } else {
                $result = wp_create_category($name);
                if (!is_wp_error($result)) {
                    $category_ids[] = $result;
                }
            }
        }
        
        return $category_ids;
    }
    
    /**
     * Extract JSON-LD script tags from content
     */
    private function extract_json_ld_scripts($content) {
        $scripts = [];
        
        // First, try to match <script type="application/ld+json">...</script> tags
        preg_match_all(
            '/<script\s+type=["\']application\/ld\+json["\']\s*>([\s\S]*?)<\/script>/i',
            $content,
            $matches,
            PREG_SET_ORDER
        );
        
        foreach ($matches as $match) {
            if (!empty($match[1])) {
                $scripts[] = trim($match[1]);
            }
        }
        
        // Fallback: If no script tags found, try to find raw JSON-LD objects
        // Look for JSON objects that start with "@context": "https://schema.org"
        if (empty($scripts)) {
            // Remove HTML tags and normalize whitespace
            $clean_content = strip_tags($content);
            $clean_content = preg_replace('/\s+/', ' ', $clean_content);
            
            // Try to find JSON-LD patterns
            // Match objects starting with "@context": "https://schema.org"
            preg_match_all(
                '/\{[^{}]*"@context"\s*:\s*"https:\/\/schema\.org"[^{}]*\}/',
                $clean_content,
                $raw_matches,
                PREG_SET_ORDER
            );
            
            // Also try multi-line JSON (common when JSON is in <p> tags with <br>)
            if (empty($raw_matches)) {
                // Remove <br> tags and normalize
                $normalized = preg_replace('/<br\s*\/?>/i', "\n", $content);
                $normalized = strip_tags($normalized);
                
                // Try to extract complete JSON objects
                preg_match_all(
                    '/\{[\s\S]*?"@context"\s*:\s*"https:\/\/schema\.org"[\s\S]*?\}/',
                    $normalized,
                    $multiline_matches,
                    PREG_SET_ORDER
                );
                
                foreach ($multiline_matches as $match) {
                    $json_str = trim($match[0]);
                    // Validate it's valid JSON
                    $decoded = json_decode($json_str, true);
                    if (json_last_error() === JSON_ERROR_NONE && isset($decoded['@context']) && $decoded['@context'] === 'https://schema.org') {
                        $scripts[] = $json_str;
                    }
                }
            } else {
                foreach ($raw_matches as $match) {
                    $json_str = trim($match[0]);
                    // Validate it's valid JSON
                    $decoded = json_decode($json_str, true);
                    if (json_last_error() === JSON_ERROR_NONE && isset($decoded['@context']) && $decoded['@context'] === 'https://schema.org') {
                        $scripts[] = $json_str;
                    }
                }
            }
        }
        
        return $scripts;
    }
    
    /**
     * Remove JSON-LD script tags from content
     */
    private function remove_json_ld_scripts($content) {
        // Remove <script type="application/ld+json">...</script> tags
        $content = preg_replace(
            '/<script\s+type=["\']application\/ld\+json["\']\s*>[\s\S]*?<\/script>/i',
            '',
            $content
        );
        
        // Also remove raw JSON-LD objects that might be in the content
        // This handles cases where JSON-LD was added without script tags
        
        // Remove JSON-LD objects that might be wrapped in <p> tags with <br> tags
        // Pattern: <p>...{...JSON-LD...}...</p>
        $content = preg_replace(
            '/<p[^>]*>[\s\S]*?\{[\s\S]*?"@context"\s*:\s*"https:\/\/schema\.org"[\s\S]*?\}[\s\S]*?<\/p>/i',
            '',
            $content
        );
        
        // Also remove standalone JSON-LD objects (not in script tags)
        // This regex handles JSON that might span multiple lines with <br> tags
        $content = preg_replace(
            '/\{[\s\S]*?"@context"\s*:\s*"https:\/\/schema\.org"[\s\S]*?\}/',
            '',
            $content
        );
        
        // Clean up any empty <p> tags that might be left behind
        $content = preg_replace('/<p[^>]*>\s*<\/p>/i', '', $content);
        
        return $content;
    }
    
    /**
     * Send webhook
     */
    private function send_webhook($event, $data) {
        $this->webhook->send($event, $data);
    }
}

