<?php
/*
Plugin Name: SEO Tool Integration
Plugin URI: https://your-seo-tool.com
Description: Connect your WordPress site to SEO Tool for automated blog publishing
Version: 1.0.0
Author: Your Company
Author URI: https://your-seo-tool.com
License: GPL v2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: seo-tool
Requires at least: 5.8
Requires PHP: 7.4
*/

// Security check
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('SEO_TOOL_VERSION', '1.0.0');
define('SEO_TOOL_PLUGIN_URL', plugin_dir_url(__FILE__));
define('SEO_TOOL_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('SEO_TOOL_PLUGIN_FILE', __FILE__);
define('SEO_TOOL_API_BASE_URL', get_option('seo_tool_api_base_url', 'https://seotoolstaging.site/api/v1'));

// Include required files
require_once SEO_TOOL_PLUGIN_PATH . 'includes/class-auth.php';
require_once SEO_TOOL_PLUGIN_PATH . 'includes/class-api.php';
require_once SEO_TOOL_PLUGIN_PATH . 'includes/class-webhook.php';
require_once SEO_TOOL_PLUGIN_PATH . 'includes/class-admin.php';
require_once SEO_TOOL_PLUGIN_PATH . 'includes/class-publisher.php';

/**
 * Initialize plugin
 */
function seo_tool_init() {
    // Initialize authentication
    $auth = new SEO_Tool_Auth();
    $auth->init();
    
    // Initialize REST API
    $api = new SEO_Tool_API();
    $api->init();
    
    // Initialize admin
    $admin = new SEO_Tool_Admin();
    $admin->init();
    
    // Inject JSON-LD into head
    add_action('wp_head', 'seo_tool_inject_json_ld', 1);
}
add_action('plugins_loaded', 'seo_tool_init');

/**
 * Inject JSON-LD structured data into head
 */
function seo_tool_inject_json_ld() {
    // Only on single post pages
    if (!is_singular('post')) {
        return;
    }
    
    global $post;
    if (!$post) {
        return;
    }
    
    // Get JSON-LD scripts from post meta
    $json_ld_scripts = get_post_meta($post->ID, '_seo_tool_json_ld', true);
    
    if (empty($json_ld_scripts) || !is_array($json_ld_scripts)) {
        return;
    }
    
    // Output each JSON-LD script in proper script tags
    foreach ($json_ld_scripts as $json_ld) {
        // Validate JSON before outputting
        $decoded = json_decode($json_ld, true);
        if (json_last_error() === JSON_ERROR_NONE) {
            echo '<script type="application/ld+json">' . "\n";
            echo wp_json_encode($decoded, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT) . "\n";
            echo '</script>' . "\n";
        }
    }
}

/**
 * Activation hook
 */
function seo_tool_activate() {
    // Create database tables if needed
    // Set default options
    add_option('seo_tool_version', SEO_TOOL_VERSION);
    add_option('seo_tool_api_base_url', 'https://seotoolstaging.site/api/v1');
    
    // Flush rewrite rules
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'seo_tool_activate');

/**
 * Deactivation hook
 */
function seo_tool_deactivate() {
    // Clean up temporary data
    flush_rewrite_rules();
}
register_deactivation_hook(__FILE__, 'seo_tool_deactivate');

/**
 * Enable WordPress debug logging for troubleshooting
 */
if (!defined('WP_DEBUG_LOG')) {
    define('WP_DEBUG_LOG', true);
}

