/**
 * SEO Tool Admin JavaScript
 */
(function($) {
    'use strict';
    
    var currentPage = 1;
    var totalPages = 1;
    var currentStatus = '';
    var blogsData = [];
    
    $(document).ready(function() {
        // Dismissible notices
        $('.notice.is-dismissible').on('click', '.notice-dismiss', function() {
            $(this).closest('.notice').fadeOut();
        });
        
        // Copy token functionality
        $('.copy-token').on('click', function(e) {
            e.preventDefault();
            var token = $(this).data('token');
            var $temp = $('<input>');
            $('body').append($temp);
            $temp.val(token).select();
            document.execCommand('copy');
            $temp.remove();
            
            var $button = $(this);
            var originalText = $button.text();
            $button.text('Copied!');
            setTimeout(function() {
                $button.text(originalText);
            }, 2000);
        });
        
        // Blogs page functionality
        if ($('.seo-tool-blogs').length && typeof seoToolAdmin !== 'undefined' && seoToolAdmin.isConnected) {
            initBlogsPage();
        }
    });
    
    function initBlogsPage() {
        fetchBlogs();
        
        $('#seo-tool-refresh-blogs').on('click', function() {
            fetchBlogs();
        });
        
        $('#seo-tool-status-filter').on('change', function() {
            currentStatus = $(this).val();
            currentPage = 1;
            fetchBlogs();
        });
        
        $('#seo-tool-prev-page').on('click', function() {
            if (currentPage > 1) {
                currentPage--;
                fetchBlogs();
            }
        });
        
        $('#seo-tool-next-page').on('click', function() {
            if (currentPage < totalPages) {
                currentPage++;
                fetchBlogs();
            }
        });
        
        // Modal close handlers
        $('.seo-tool-modal-close').on('click', function() {
            closeModal();
        });
        
        $('#seo-tool-blog-modal').on('click', function(e) {
            if ($(e.target).is('#seo-tool-blog-modal')) {
                closeModal();
            }
        });
        
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape') {
                closeModal();
            }
        });
    }
    
    function fetchBlogs() {
        showLoading();
        
        $.ajax({
            url: seoToolAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'seo_tool_fetch_blogs',
                nonce: seoToolAdmin.nonce,
                page: currentPage,
                status: currentStatus
            },
            success: function(response) {
                hideLoading();
                
                if (response.success) {
                    blogsData = response.data.blogs || [];
                    var pagination = response.data.pagination || {};
                    
                    totalPages = pagination.totalPages || 1;
                    currentPage = pagination.page || 1;
                    
                    updateBlogsCount(pagination.total || 0);
                    renderBlogs(blogsData);
                    updatePagination();
                } else {
                    showError(response.data.message || 'Failed to fetch blogs');
                }
            },
            error: function(xhr, status, error) {
                hideLoading();
                showError('Network error: ' + error);
            }
        });
    }
    
    function renderBlogs(blogs) {
        var $table = $('#seo-tool-blogs-table');
        var $body = $('#seo-tool-blogs-body');
        var $empty = $('#seo-tool-blogs-empty');
        
        if (!blogs || blogs.length === 0) {
            $table.hide();
            $empty.show();
            return;
        }
        
        $empty.hide();
        $body.empty();
        
        blogs.forEach(function(blog, index) {
            var statusClass = getStatusClass(blog.status);
            var seoScore = blog.seoScore || 0;
            var scoreClass = getScoreClass(seoScore);
            var excerpt = blog.excerpt || '';
            if (excerpt.length > 100) {
                excerpt = excerpt.substring(0, 100) + '...';
            }
            
            var publishDate = blog.publishDate ? formatDate(blog.publishDate) : 'Not set';
            if (blog.publishTime) {
                publishDate += ' ' + blog.publishTime;
            }
            
            var row = '<tr data-index="' + index + '">' +
                '<td class="column-title">' +
                    '<div class="blog-title">' + escapeHtml(blog.title || 'Untitled') + '</div>' +
                    '<div class="blog-excerpt">' + escapeHtml(excerpt) + '</div>' +
                '</td>' +
                '<td class="column-status">' +
                    '<span class="seo-tool-status seo-tool-status-' + statusClass + '">' + (blog.status || 'Draft') + '</span>' +
                '</td>' +
                '<td class="column-date">' + publishDate + '</td>' +
                '<td class="column-seo">' +
                    '<div class="seo-tool-seo-score">' +
                        '<div class="score-bar"><div class="score-fill score-' + scoreClass + '" style="width: ' + seoScore + '%"></div></div>' +
                        '<span>' + seoScore + '%</span>' +
                    '</div>' +
                '</td>' +
                '<td class="column-actions">' +
                    '<div class="seo-tool-actions">' +
                        '<button type="button" class="button seo-tool-preview-btn" data-index="' + index + '">Preview</button>' +
                        '<button type="button" class="button button-primary seo-tool-publish-btn" data-index="' + index + '">Publish</button>' +
                    '</div>' +
                '</td>' +
            '</tr>';
            
            $body.append(row);
        });
        
        $table.show();
        
        // Bind action buttons
        $('.seo-tool-preview-btn').on('click', function() {
            var index = $(this).data('index');
            openPreviewModal(blogsData[index]);
        });
        
        $('.seo-tool-publish-btn').on('click', function() {
            var index = $(this).data('index');
            var $btn = $(this);
            publishBlog(blogsData[index], $btn);
        });
    }
    
    function openPreviewModal(blog) {
        if (!blog) return;
        
        $('#seo-tool-modal-title').text(blog.title || 'Untitled');
        
        var metaHtml = '';
        if (blog.status) {
            metaHtml += '<div><strong>Status:</strong> ' + blog.status + '</div>';
        }
        if (blog.publishDate) {
            metaHtml += '<div><strong>Publish Date:</strong> ' + formatDate(blog.publishDate) + '</div>';
        }
        if (blog.seoScore) {
            metaHtml += '<div><strong>SEO Score:</strong> ' + blog.seoScore + '%</div>';
        }
        if (blog.meta && blog.meta.focusKeyword) {
            metaHtml += '<div><strong>Focus Keyword:</strong> ' + escapeHtml(blog.meta.focusKeyword) + '</div>';
        }
        $('#seo-tool-modal-meta').html(metaHtml);
        
        $('#seo-tool-modal-content').html(blog.content || '<p>No content available.</p>');
        
        // Store blog data for publish action
        $('#seo-tool-modal-publish').data('blog', blog);
        
        $('#seo-tool-blog-modal').show();
        $('body').css('overflow', 'hidden');
    }
    
    function closeModal() {
        $('#seo-tool-blog-modal').hide();
        $('body').css('overflow', '');
    }
    
    function publishBlog(blog, $btn) {
        if (!blog) return;
        
        var originalText = $btn ? $btn.text() : '';
        if ($btn) {
            $btn.text('Publishing...').prop('disabled', true);
        }
        
        $.ajax({
            url: seoToolAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'seo_tool_publish_blog',
                nonce: seoToolAdmin.nonce,
                blog_id: blog.id,
                title: blog.title,
                content: blog.content,
                excerpt: blog.excerpt || '',
                slug: blog.slug || '',
                status: blog.status || 'DRAFT'
            },
            success: function(response) {
                if ($btn) {
                    $btn.prop('disabled', false);
                }
                
                if (response.success) {
                    if ($btn) {
                        $btn.text('Published!').addClass('button-success');
                        setTimeout(function() {
                            $btn.text(originalText).removeClass('button-success');
                        }, 2000);
                    }
                    
                    closeModal();
                    
                    // Show success notification
                    alert('Blog published successfully!\n\nView: ' + response.data.post_url);
                } else {
                    if ($btn) {
                        $btn.text(originalText);
                    }
                    alert('Error: ' + (response.data.message || 'Failed to publish'));
                }
            },
            error: function(xhr, status, error) {
                if ($btn) {
                    $btn.text(originalText).prop('disabled', false);
                }
                alert('Network error: ' + error);
            }
        });
    }
    
    // Modal publish button handler
    $(document).on('click', '#seo-tool-modal-publish', function() {
        var blog = $(this).data('blog');
        if (blog) {
            publishBlog(blog, $(this));
        }
    });
    
    function showLoading() {
        $('#seo-tool-blogs-loading').show();
        $('#seo-tool-blogs-table').hide();
        $('#seo-tool-blogs-empty').hide();
        $('#seo-tool-blogs-error').hide();
    }
    
    function hideLoading() {
        $('#seo-tool-blogs-loading').hide();
    }
    
    function showError(message) {
        $('#seo-tool-blogs-error').show().find('p').text(message);
        $('#seo-tool-blogs-table').hide();
        $('#seo-tool-blogs-empty').hide();
    }
    
    function updateBlogsCount(count) {
        $('#seo-tool-blogs-count').text(count + ' blog post' + (count !== 1 ? 's' : '') + ' found');
    }
    
    function updatePagination() {
        var $pagination = $('#seo-tool-pagination');
        var $prev = $('#seo-tool-prev-page');
        var $next = $('#seo-tool-next-page');
        var $info = $('#seo-tool-page-info');
        
        if (totalPages > 1) {
            $pagination.show();
            $info.text('Page ' + currentPage + ' of ' + totalPages);
            $prev.prop('disabled', currentPage <= 1);
            $next.prop('disabled', currentPage >= totalPages);
        } else {
            $pagination.hide();
        }
    }
    
    function getStatusClass(status) {
        if (!status) return 'draft';
        status = status.toLowerCase();
        if (status === 'publish' || status === 'published') return 'publish';
        if (status === 'scheduled') return 'scheduled';
        return 'draft';
    }
    
    function getScoreClass(score) {
        if (score >= 70) return 'high';
        if (score >= 40) return 'medium';
        return 'low';
    }
    
    function formatDate(dateStr) {
        if (!dateStr) return '';
        var date = new Date(dateStr);
        if (isNaN(date.getTime())) return dateStr;
        
        var options = { year: 'numeric', month: 'short', day: 'numeric' };
        return date.toLocaleDateString('en-US', options);
    }
    
    function escapeHtml(text) {
        if (!text) return '';
        var div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
    
})(jQuery);
