<?php
/**
 * Admin Settings Page
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Tool_Admin {
    private $auth;
    
    public function __construct() {
        $this->auth = new SEO_Tool_Auth();
    }
    
    public function init() {
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_init', [$this, 'handle_disconnect']);
        add_action('admin_init', [$this, 'handle_save_key']);
        add_action('wp_ajax_seo_tool_fetch_blogs', [$this, 'ajax_fetch_blogs']);
        add_action('wp_ajax_seo_tool_publish_blog', [$this, 'ajax_publish_blog']);
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_menu_page(
            'Uplift AI',
            'Uplift AI',
            'manage_options',
            'seo-tool',
            [$this, 'render_settings_page'],
            'dashicons-admin-links',
            30
        );
        
        add_submenu_page(
            'seo-tool',
            'Settings',
            'Settings',
            'manage_options',
            'seo-tool',
            [$this, 'render_settings_page']
        );
        
        add_submenu_page(
            'seo-tool',
            'Blog Posts',
            'Blog Posts',
            'manage_options',
            'seo-tool-blogs',
            [$this, 'render_blogs_page']
        );
    }
    
    /**
     * Enqueue scripts and styles
     */
    public function enqueue_scripts($hook) {
        $allowed_hooks = ['toplevel_page_seo-tool', 'uplift-ai_page_seo-tool-blogs'];
        
        if (!in_array($hook, $allowed_hooks)) {
            return;
        }
        
        wp_enqueue_style(
            'seo-tool-admin',
            SEO_TOOL_PLUGIN_URL . 'admin/assets/css/admin.css',
            [],
            SEO_TOOL_VERSION
        );
        
        wp_enqueue_script(
            'seo-tool-admin',
            SEO_TOOL_PLUGIN_URL . 'admin/assets/js/admin.js',
            ['jquery'],
            SEO_TOOL_VERSION,
            true
        );
        
        wp_localize_script('seo-tool-admin', 'seoToolAdmin', [
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('seo_tool_ajax'),
            'isConnected' => $this->auth->is_connected(),
        ]);
    }
    
    /**
     * Handle disconnect
     */
    public function handle_disconnect() {
        if (!isset($_POST['seo_tool_disconnect']) || !check_admin_referer('seo_tool_disconnect')) {
            return;
        }
        
        $this->auth->disconnect();
        wp_redirect(admin_url('admin.php?page=seo-tool&disconnected=1'));
        exit;
    }
    
    
    /**
     * Handle integration key save
     */
    public function handle_save_key() {
        if (!isset($_POST['seo_tool_save_key']) || !check_admin_referer('seo_tool_save_key')) {
            return;
        }
        
        $key = isset($_POST['integration_key']) ? sanitize_text_field($_POST['integration_key']) : '';
        $post_mode = isset($_POST['post_mode']) ? sanitize_text_field($_POST['post_mode']) : 'publish';
        
        if (empty($key)) {
            wp_redirect(admin_url('admin.php?page=seo-tool&error=empty_key'));
            exit;
        }
        
        $result = $this->auth->validate_integration_key($key);
        
        // Log the result for debugging
        error_log('[SEO Tool] Validation result: ' . print_r($result, true));
        
        if ($result['success']) {
            // Double-check that options are set (they should already be set in validate_integration_key)
            update_option('seo_tool_integration_key', $key);
            update_option('seo_tool_post_mode', $post_mode);
            update_option('seo_tool_connected', true);
            update_option('seo_tool_connected_at', current_time('mysql'));
            if (isset($result['user_id'])) {
                update_option('seo_tool_user_id', $result['user_id']);
            }
            
            // Verify connection status is set
            $connected_status = get_option('seo_tool_connected', false);
            $stored_key = get_option('seo_tool_integration_key', '');
            error_log('[SEO Tool] After validation - connected: ' . ($connected_status ? 'true' : 'false') . ', key: ' . (!empty($stored_key) ? 'set' : 'empty'));
            
            wp_redirect(admin_url('admin.php?page=seo-tool&connected=1'));
            exit;
        } else {
            wp_redirect(admin_url('admin.php?page=seo-tool&error=' . urlencode($result['message'])));
            exit;
        }
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        $is_connected = $this->auth->is_connected();
        $connected_at = get_option('seo_tool_connected_at');
        $user_id = get_option('seo_tool_user_id');
        $last_publish = get_option('seo_tool_last_publish');
        $total_published = get_option('seo_tool_total_published', 0);
        $integration_key = get_option('seo_tool_integration_key', '');
        $post_mode = get_option('seo_tool_post_mode', 'publish');
        $error_message = isset($_GET['error']) ? urldecode($_GET['error']) : '';
        
        ?>
        <div class="wrap seo-tool-settings">
            <h1>Uplift AI Integration</h1>
            
            <?php if (isset($_GET['connected'])): ?>
                <div class="notice notice-success is-dismissible">
                    <p>Successfully connected to Uplift AI!</p>
                </div>
            <?php endif; ?>
            
            <?php if (isset($_GET['disconnected'])): ?>
                <div class="notice notice-info is-dismissible">
                    <p>Disconnected from Uplift AI.</p>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo esc_html($error_message); ?></p>
                </div>
            <?php endif; ?>
            
            <?php if (!$is_connected): ?>
                <div class="card">
                    <h2>Connect to Uplift AI</h2>
                    
                    <!-- Integration Key Method -->
                    <div style="margin-bottom: 30px;">
                        <h3>Method 1: Integration Key (Recommended)</h3>
                        <p>Generate an integration key in your Uplift AI dashboard and paste it here.</p>
                        
                        <form method="post" action="">
                            <?php wp_nonce_field('seo_tool_save_key'); ?>
                            <table class="form-table">
                                <tr>
                                    <th>
                                        <label for="integration_key">Integration Key</label>
                                    </th>
                                    <td>
                                        <input 
                                            type="text" 
                                            id="integration_key" 
                                            name="integration_key" 
                                            value="<?php echo esc_attr($integration_key); ?>"
                                            class="regular-text"
                                            placeholder="wp_key_xxxx-xxxx-xxxx"
                                        />
                                        <p class="description">
                                            Generate this key in <strong>Uplift AI</strong> when creating your WordPress integration.
                                        </p>
                                    </td>
                                </tr>
                                <tr>
                                    <th>
                                        <label for="post_mode">Post Mode</label>
                                    </th>
                                    <td>
                                        <select id="post_mode" name="post_mode" class="regular-text">
                                            <option value="publish" <?php selected($post_mode, 'publish'); ?>>Publish Directly</option>
                                            <option value="draft" <?php selected($post_mode, 'draft'); ?>>Save as Draft</option>
                                        </select>
                                        <p class="description">
                                            Choose whether incoming posts are published immediately or saved as drafts.
                                        </p>
                                    </td>
                                </tr>
                            </table>
                            <p class="submit">
                                <input type="submit" name="seo_tool_save_key" class="button button-primary" value="Save" />
                            </p>
                        </form>
                    </div>
                </div>
            <?php else: ?>
                <div class="card">
                    <h2>✅ Connected</h2>
                    <p>Your WordPress site is connected to your Uplift AI account.</p>
                    
                    <table class="form-table">
                        <tr>
                            <th>Status</th>
                            <td>
                                <span style="color: green;">●</span> Connected
                                <br>
                                <small>Connected at: <?php echo esc_html($connected_at); ?></small>
                            </td>
                        </tr>
                        <tr>
                            <th>Account</th>
                            <td>User ID: <?php echo esc_html($user_id); ?></td>
                        </tr>
                        <tr>
                            <th>Statistics</th>
                            <td>
                                Total published: <strong><?php echo esc_html($total_published); ?></strong>
                                <?php if ($last_publish): ?>
                                    <br>
                                    Last publish: <?php echo esc_html($last_publish); ?>
                                <?php endif; ?>
                            </td>
                        </tr>
                    </table>
                    
                    <form method="post" action="">
                        <?php wp_nonce_field('seo_tool_disconnect'); ?>
                        <input type="hidden" name="seo_tool_disconnect" value="1">
                        <button type="submit" class="button button-secondary" onclick="return confirm('Are you sure you want to disconnect?');">
                            Disconnect
                        </button>
                    </form>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
    
    /**
     * Render blogs page
     */
    public function render_blogs_page() {
        $is_connected = $this->auth->is_connected();
        $integration_key = get_option('seo_tool_integration_key', '');
        
        ?>
        <div class="wrap seo-tool-settings seo-tool-blogs">
            <h1>Blog Posts from Uplift AI</h1>
            
            <?php if (!$is_connected): ?>
                <div class="notice notice-warning">
                    <p>Please <a href="<?php echo admin_url('admin.php?page=seo-tool'); ?>">connect to Uplift AI</a> first to view your blog posts.</p>
                </div>
            <?php else: ?>
                <div class="card">
                    <div class="seo-tool-blogs-header">
                        <div class="seo-tool-blogs-actions">
                            <button type="button" id="seo-tool-refresh-blogs" class="button button-secondary">
                                <span class="dashicons dashicons-update"></span> Refresh
                            </button>
                            <select id="seo-tool-status-filter" class="seo-tool-filter">
                                <option value="">All Status</option>
                                <option value="PUBLISH">Published</option>
                                <option value="DRAFT">Draft</option>
                            </select>
                        </div>
                        <div class="seo-tool-blogs-info">
                            <span id="seo-tool-blogs-count">Loading...</span>
                        </div>
                    </div>
                    
                    <div id="seo-tool-blogs-loading" class="seo-tool-loading">
                        <span class="spinner is-active"></span>
                        <span>Loading blog posts...</span>
                    </div>
                    
                    <div id="seo-tool-blogs-error" class="seo-tool-error" style="display: none;">
                        <p></p>
                    </div>
                    
                    <table id="seo-tool-blogs-table" class="wp-list-table widefat fixed striped" style="display: none;">
                        <thead>
                            <tr>
                                <th class="column-title">Title</th>
                                <th class="column-status">Status</th>
                                <th class="column-date">Publish Date</th>
                                <th class="column-seo">SEO Score</th>
                                <th class="column-actions">Actions</th>
                            </tr>
                        </thead>
                        <tbody id="seo-tool-blogs-body">
                        </tbody>
                    </table>
                    
                    <div id="seo-tool-blogs-empty" class="seo-tool-empty" style="display: none;">
                        <p>No blog posts found. Create some content in Uplift AI to see it here.</p>
                    </div>
                    
                    <div id="seo-tool-pagination" class="seo-tool-pagination" style="display: none;">
                        <button type="button" id="seo-tool-prev-page" class="button" disabled>&laquo; Previous</button>
                        <span id="seo-tool-page-info">Page 1 of 1</span>
                        <button type="button" id="seo-tool-next-page" class="button">Next &raquo;</button>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        
        <!-- Blog Preview Modal -->
        <div id="seo-tool-blog-modal" class="seo-tool-modal" style="display: none;">
            <div class="seo-tool-modal-content">
                <div class="seo-tool-modal-header">
                    <h2 id="seo-tool-modal-title"></h2>
                    <button type="button" class="seo-tool-modal-close">&times;</button>
                </div>
                <div class="seo-tool-modal-body">
                    <div id="seo-tool-modal-meta" class="seo-tool-modal-meta"></div>
                    <div id="seo-tool-modal-content"></div>
                </div>
                <div class="seo-tool-modal-footer">
                    <button type="button" id="seo-tool-modal-publish" class="button button-primary">Publish to WordPress</button>
                    <button type="button" class="button seo-tool-modal-close">Close</button>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * AJAX handler: Fetch blogs from API
     */
    public function ajax_fetch_blogs() {
        check_ajax_referer('seo_tool_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }
        
        $integration_key = get_option('seo_tool_integration_key', '');
        
        if (empty($integration_key)) {
            wp_send_json_error(['message' => 'Not connected to Uplift AI']);
            return;
        }
        
        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : '';
        $limit = 10;
        
        $api_base_url = preg_replace('/\/api\/v1\/?$/', '', SEO_TOOL_API_BASE_URL);
        
        $url = $api_base_url . '/api/public/v1/blogs/' . urlencode($integration_key);
        $url .= '?page=' . $page . '&limit=' . $limit;
        
        if (!empty($status)) {
            $url .= '&status=' . urlencode($status);
        }
        
        $response = wp_remote_get($url, [
            'timeout' => 30,
            'headers' => [
                'Content-Type' => 'application/json',
            ],
        ]);
        
        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Failed to fetch blogs: ' . $response->get_error_message()]);
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!$data || !isset($data['success']) || !$data['success']) {
            $error_message = isset($data['error']) ? $data['error'] : 'Unknown error';
            wp_send_json_error(['message' => $error_message]);
            return;
        }
        
        wp_send_json_success($data['data']);
    }
    
    /**
     * AJAX handler: Publish a blog to WordPress
     */
    public function ajax_publish_blog() {
        check_ajax_referer('seo_tool_ajax', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => 'Unauthorized']);
            return;
        }
        
        $blog_id = isset($_POST['blog_id']) ? sanitize_text_field($_POST['blog_id']) : '';
        $title = isset($_POST['title']) ? sanitize_text_field($_POST['title']) : '';
        $content = isset($_POST['content']) ? wp_kses_post($_POST['content']) : '';
        $excerpt = isset($_POST['excerpt']) ? sanitize_textarea_field($_POST['excerpt']) : '';
        $slug = isset($_POST['slug']) ? sanitize_title($_POST['slug']) : '';
        $status = isset($_POST['status']) ? sanitize_text_field($_POST['status']) : 'draft';
        
        if (empty($title) || empty($content)) {
            wp_send_json_error(['message' => 'Title and content are required']);
            return;
        }
        
        $post_status = ($status === 'PUBLISH' || $status === 'publish') ? 'publish' : 'draft';
        $post_mode = get_option('seo_tool_post_mode', 'publish');
        if ($post_mode === 'draft') {
            $post_status = 'draft';
        }
        
        $existing_post = get_posts([
            'meta_key' => '_seo_tool_blog_id',
            'meta_value' => $blog_id,
            'post_type' => 'post',
            'posts_per_page' => 1,
        ]);
        
        $post_data = [
            'post_title' => $title,
            'post_content' => $content,
            'post_excerpt' => $excerpt,
            'post_name' => $slug,
            'post_status' => $post_status,
            'post_type' => 'post',
        ];
        
        if (!empty($existing_post)) {
            $post_data['ID'] = $existing_post[0]->ID;
            $post_id = wp_update_post($post_data, true);
            $action = 'updated';
        } else {
            $post_id = wp_insert_post($post_data, true);
            $action = 'created';
        }
        
        if (is_wp_error($post_id)) {
            wp_send_json_error(['message' => 'Failed to create post: ' . $post_id->get_error_message()]);
            return;
        }
        
        update_post_meta($post_id, '_seo_tool_blog_id', $blog_id);
        update_post_meta($post_id, '_seo_tool_synced_at', current_time('mysql'));
        
        $total_published = get_option('seo_tool_total_published', 0);
        update_option('seo_tool_total_published', $total_published + 1);
        update_option('seo_tool_last_publish', current_time('mysql'));
        
        wp_send_json_success([
            'message' => 'Post ' . $action . ' successfully',
            'post_id' => $post_id,
            'post_url' => get_permalink($post_id),
            'edit_url' => get_edit_post_link($post_id, 'raw'),
        ]);
    }
}

