<?php
/**
 * Custom REST API Endpoints
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Tool_API {
    private $auth;
    
    public function __construct() {
        $this->auth = new SEO_Tool_Auth();
    }
    
    public function init() {
        add_action('rest_api_init', [$this, 'register_routes']);
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes() {
        register_rest_route('seo-tool/v1', '/publish', [
            'methods' => 'POST',
            'callback' => [$this, 'publish_post'],
            'permission_callback' => [$this, 'verify_token'],
        ]);
        
        register_rest_route('seo-tool/v1', '/status', [
            'methods' => 'GET',
            'callback' => [$this, 'get_status'],
            'permission_callback' => [$this, 'verify_token'],
        ]);
        
        register_rest_route('seo-tool/v1', '/test', [
            'methods' => 'GET',
            'callback' => [$this, 'test_connection'],
            'permission_callback' => [$this, 'verify_token'],
        ]);
    }
    
    /**
     * Verify integration key
     */
    public function verify_token($request) {
        $auth_header = $request->get_header('Authorization');
        
        if (!$auth_header) {
            error_log('[SEO Tool API] Missing Authorization header');
            return new WP_Error('missing_token', 'Authorization header required', ['status' => 401]);
        }
        
        // Extract Bearer token (integration key)
        if (preg_match('/Bearer\s+(.*)$/i', $auth_header, $matches)) {
            $token = trim($matches[1]); // Trim whitespace
        } else {
            error_log('[SEO Tool API] Invalid token format in Authorization header');
            return new WP_Error('invalid_token', 'Invalid token format', ['status' => 401]);
        }
        
        // Get stored key for comparison
        $stored_key = $this->auth->get_integration_key();
        
        // Log for debugging (only first 10 chars for security)
        error_log('[SEO Tool API] Verifying token. Received: ' . substr($token, 0, 10) . '..., Stored: ' . (!empty($stored_key) ? substr($stored_key, 0, 10) . '...' : 'empty'));
        
        // Verify integration key
        if (!$this->auth->verify_token($token)) {
            error_log('[SEO Tool API] Token verification failed. Token length: ' . strlen($token) . ', Stored key length: ' . strlen($stored_key ?: ''));
            return new WP_Error('invalid_token', 'Invalid or expired integration key', ['status' => 401]);
        }
        
        error_log('[SEO Tool API] Token verified successfully');
        return true;
    }
    
    /**
     * Publish post endpoint
     */
    public function publish_post($request) {
        $publisher = new SEO_Tool_Publisher();
        return $publisher->publish($request->get_json_params());
    }
    
    /**
     * Get status endpoint
     */
    public function get_status($request) {
        return [
            'plugin_version' => SEO_TOOL_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'api_key_active' => $this->auth->is_connected(),
            'connection_status' => $this->auth->is_connected() ? 'connected' : 'disconnected',
            'last_publish' => get_option('seo_tool_last_publish'),
            'total_published' => (int) get_option('seo_tool_total_published', 0),
        ];
    }
    
    /**
     * Test connection endpoint
     */
    public function test_connection($request) {
        return [
            'success' => true,
            'message' => 'Connection successful',
            'timestamp' => current_time('c'),
        ];
    }
}

