<?php
/**
 * Integration Key Authentication Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class SEO_Tool_Auth {
    private $api_base_url;
    
    public function __construct() {
        $this->api_base_url = SEO_TOOL_API_BASE_URL;
    }
    
    /**
     * Get API base URL
     */
    public function get_api_base_url() {
        return $this->api_base_url;
    }
    
    public function init() {
        // No OAuth callbacks needed
    }
    
    /**
     * Disconnect from SEO Tool
     */
    public function disconnect() {
        delete_option('seo_tool_integration_key');
        delete_option('seo_tool_user_id');
        delete_option('seo_tool_connected');
        delete_option('seo_tool_connected_at');
        delete_option('seo_tool_webhook_url');
    }
    
    /**
     * Check if connected
     */
    public function is_connected() {
        $connected = get_option('seo_tool_connected', false);
        $integration_key = get_option('seo_tool_integration_key');
        return $connected && !empty($integration_key);
    }
    
    /**
     * Get integration key
     */
    public function get_integration_key() {
        return get_option('seo_tool_integration_key');
    }
    
    /**
     * Verify integration key for API requests
     */
    public function verify_token($token) {
        $stored_key = $this->get_integration_key();
        
        // Trim both keys for comparison
        $token = trim($token);
        $stored_key = trim($stored_key ?: '');
        
        // Log for debugging (only if keys don't match)
        if ($token !== $stored_key) {
            error_log('[SEO Tool Auth] Key mismatch. Token: ' . substr($token, 0, 10) . '... (len: ' . strlen($token) . '), Stored: ' . substr($stored_key, 0, 10) . '... (len: ' . strlen($stored_key) . ')');
        }
        
        return !empty($stored_key) && $token === $stored_key;
    }
    
    /**
     * Validate integration key and connect
     */
    public function validate_integration_key($integration_key) {
        $url = $this->api_base_url . '/auth/wordpress/validate-key';
        
        // Log the request for debugging
        error_log('[SEO Tool] Validating key: ' . substr($integration_key, 0, 20) . '...');
        error_log('[SEO Tool] API URL: ' . $url);
        error_log('[SEO Tool] WordPress Site URL: ' . home_url());
        
        $response = wp_remote_post($url, [
            'body' => json_encode([
                'integrationKey' => $integration_key,
                'wordpressSiteUrl' => home_url(),
            ]),
            'headers' => [
                'Content-Type' => 'application/json'
            ],
            'timeout' => 30,
            'sslverify' => true // Enable SSL verification for production
        ]);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            error_log('[SEO Tool] WP Error: ' . $error_message);
            return [
                'success' => false,
                'message' => 'Connection failed: ' . $error_message . '. Check if backend is running at ' . $this->api_base_url
            ];
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        error_log('[SEO Tool] Response Code: ' . $response_code);
        error_log('[SEO Tool] Response Body: ' . substr($response_body, 0, 500));
        
        // Handle 401 Unauthorized - Invalid integration key
        if ($response_code === 401) {
            $body = json_decode($response_body, true);
            $error_msg = 'Invalid integration key. Please check your integration key and try again.';
            
            if ($body && isset($body['message'])) {
                $error_msg = $body['message'];
            } elseif ($body && isset($body['error'])) {
                if (is_array($body['error'])) {
                    $error_msg = $body['error']['message'] ?? 'Invalid integration key. Please check your integration key and try again.';
                } else {
                    $error_msg = $body['error'];
                }
            }
            
            error_log('[SEO Tool] 401 Unauthorized: ' . $error_msg);
            
            return [
                'success' => false,
                'message' => $error_msg
            ];
        }
        
        // Handle other error codes
        if ($response_code !== 200) {
            $body = json_decode($response_body, true);
            $error_msg = 'Backend returned error ' . $response_code . '.';
            
            if ($body && isset($body['message'])) {
                $error_msg = $body['message'];
            } elseif ($body && isset($body['error'])) {
                if (is_array($body['error'])) {
                    $error_msg = $body['error']['message'] ?? $error_msg;
                } else {
                    $error_msg = $body['error'];
                }
            }
            
            return [
                'success' => false,
                'message' => $error_msg . ' Check backend logs for more details.'
            ];
        }
        
        $body = json_decode($response_body, true);
        
        if (!$body) {
            error_log('[SEO Tool] Failed to parse JSON response');
            return [
                'success' => false,
                'message' => 'Invalid response from backend. Response: ' . substr($response_body, 0, 200)
            ];
        }
        
        // Log the full response for debugging
        error_log('[SEO Tool] Response body: ' . print_r($body, true));
        
        // Check for success in data object (standard response format)
        if (isset($body['data']['user_id']) || isset($body['data']['verified'])) {
            // Store integration key and connection status
            update_option('seo_tool_integration_key', $integration_key);
            update_option('seo_tool_user_id', $body['data']['user_id'] ?? null);
            update_option('seo_tool_connected', true);
            update_option('seo_tool_connected_at', current_time('mysql'));
            
            error_log('[SEO Tool] ✅ Key validated successfully (via data object)');
            
            return [
                'success' => true,
                'user_id' => $body['data']['user_id'] ?? null
            ];
        }
        
        // Also check for direct user_id in response (backward compatibility)
        if (isset($body['user_id'])) {
            update_option('seo_tool_integration_key', $integration_key);
            update_option('seo_tool_user_id', $body['user_id']);
            update_option('seo_tool_connected', true);
            update_option('seo_tool_connected_at', current_time('mysql'));
            
            error_log('[SEO Tool] ✅ Key validated successfully (via direct user_id)');
            
            return [
                'success' => true,
                'user_id' => $body['user_id']
            ];
        }
        
        // Check if response indicates success (even without explicit user_id)
        if (isset($body['success']) && $body['success'] === true) {
            // If we have verified flag or any data, consider it successful
            if (isset($body['data']['verified']) || isset($body['data']['user_id']) || isset($body['verified'])) {
                update_option('seo_tool_integration_key', $integration_key);
                update_option('seo_tool_user_id', $body['data']['user_id'] ?? $body['user_id'] ?? null);
                update_option('seo_tool_connected', true);
                update_option('seo_tool_connected_at', current_time('mysql'));
                
                error_log('[SEO Tool] ✅ Key validated successfully (via success flag)');
                
                return [
                    'success' => true,
                    'user_id' => $body['data']['user_id'] ?? $body['user_id'] ?? null
                ];
            }
        }
        
        // Check for error message in response
        $error_msg = $body['message'] ?? 'Failed to validate integration key';
        if (isset($body['error'])) {
            if (is_array($body['error'])) {
                $error_msg = $body['error']['message'] ?? $error_msg;
            } else {
                $error_msg .= ': ' . $body['error'];
            }
        }
        
        error_log('[SEO Tool] Validation failed: ' . $error_msg);
        error_log('[SEO Tool] Full response structure: ' . print_r($body, true));
        
        return [
            'success' => false,
            'message' => $error_msg
        ];
    }
}

